﻿using System;

namespace rglib.libtest {
    enum E_API_ERROR_CODES : uint
    {
        /// <summary>
        /// Все хорошо
        /// </summary>
        EC_OK = 0,
        /// <summary>
        /// Все плохо
        /// </summary>
        EC_FAIL,
        /// <summary>
        /// Функциональность не реализована
        /// </summary>
        EC_NOT_IMPLEMENTED,
        /// <summary>
        /// Неверно задан один из аргументов
        /// </summary>
        EC_BAD_ARGUMENT,
        /// <summary>
        /// Указан невалидный дескриптор ресурса, или ресурс закрыт
        /// </summary>
        EC_INVALID_HANDLE,
        /// <summary>
        /// Указан дескриптор несовместимого ресурса
        /// </summary>
        EC_INVALID_RESOURCE,
        /// <summary>
        /// Указан неподдерживаемый тип соединения
        /// </summary>
        EC_INVALID_CONNECTION_TYPE,
        /// <summary>
        /// Адрес подключения не соответствует типу подключения
        /// </summary>
        EC_INVALID_CONNECTION_ADDRESS,
        /// <summary>
        /// Указанный аддрес устройства является недопустимым для указанного типа подключения
        /// </summary>
        EC_INVALID_DEVICE_ADDRESS,
        /// <summary>
        /// Функциональность не поддерживается целевым устройством
        /// </summary>
        EC_DEVICE_OPERATION_UNSUPPORTED,
        /// <summary>
        /// Устройство было отключено в процессе обмена (для USB устройств).
        /// </summary>
        EC_DEVICE_NOT_CONNECTED,
        /// <summary>
        /// Устройство не отвечает на запросы
        /// </summary>
        EC_DEVICE_NO_RESPOND,
        /// <summary>
        /// Ошибка обмена данными с устройством
        /// </summary>
        EC_DEVICE_COMM_FAILURE,
        /// <summary>
        /// Ошибка при обработке входящих сообщений протокола
        /// </summary>
        EC_DEVICE_PROTOCOL_FAILURE,
        /// <summary>
        /// Очередь событий пуста
        /// </summary>
        EC_POLL_NO_EVENTS,
        /// <summary>
        /// Очередь закрыта или уничтожена
        /// </summary>
        EC_POLL_QUEUE_CLOSED,

        // ошибки взаимодействия
        /// <summary>
        /// Устройству требуется инициализация
        /// </summary>
        EC_CALL_INIT,


        /// <summary>
        /// Нет такой команды или команда не поддерживается устройством
        /// </summary>
        EC_DEVICE_INVALID_COMMAND,

        /// <summary>
        /// указан недопустимый параметр команды
        /// </summary>
        EC_DEVICE_INVALID_PARAM,

        /// <summary>
        /// Неверный PIN для команд настройки
        /// </summary>
        EC_DEVICE_INVALID_PIN,

        /// <summary>
        /// Истекло время ожидания на получение устройством нужной команды
        /// </summary>
        EC_DEVICE_COMMAND_TIMEOUT,

        /// <summary>
        /// Нет карты на считывателе
        /// </summary>
        EC_DEVICE_NO_CARD,

        /// <summary>
        /// Устройство не смогло распознать карту
        /// </summary>
        EC_DEVICE_UNKNOWN_CARD,

        /// <summary>
        /// Карта несовместима с выполняемой операцией
        /// </summary>
        EC_DEVICE_INCOMPATIBLE_CARD,

        /// <summary>
        /// Авторизация карты не проходит
        /// </summary>
        EC_DEVICE_AUTH_FAIL,

        /// <summary>
        /// Используется неподходящий профиль
        /// </summary>
        EC_DEVICE_PROFILE_FAIL,

        /// <summary>
        /// Авторизация карты не проходит
        /// </summary>
        EC_DEVICE_RW_FAIL,


        /// <summary>
        /// Ошибка при открытии соединения
        /// </summary>
        EC_IO_OPEN_FAIL,
        /// <summary>
        /// Ошибка при закрытии соединения
        /// </summary>
        EC_IO_CLOSE_FAIL,
        /// <summary>
        /// Ошибка при откправке данных
        /// </summary>
        EC_IO_READ_FAIL,

        /// <summary>
        /// Ошибка при чтении данных
        /// </summary>
        EC_IO_WRITE_FAIL,

        /// <summary>
        /// Операция прервана, соединение было закрыто
        /// </summary>
        EC_IO_CLOSED


    }

    public static class ApiErrorHelper {
        /// <summary>
        /// Gets the error description.
        /// </summary>
        /// <param name="errorCode">The error code.</param>
        /// <returns></returns>
        /// <exception cref="System.ArgumentOutOfRangeException">errorCode - null</exception>
        public static string GetErrorDescription(uint errorCode) {
            if (Enum.IsDefined(typeof(E_API_ERROR_CODES), errorCode)) {
                switch ((E_API_ERROR_CODES) errorCode) {
                    case E_API_ERROR_CODES.EC_OK:
                        return "Все хорошо";
                    case E_API_ERROR_CODES.EC_FAIL:
                        return "Все плохо";
                    case E_API_ERROR_CODES.EC_NOT_IMPLEMENTED:
                        return "Функциональность не реализована";
                    case E_API_ERROR_CODES.EC_BAD_ARGUMENT:
                        return "Неверно задан один из аргументов";
                    case E_API_ERROR_CODES.EC_INVALID_HANDLE:
                        return "Указан невалидный дескриптор ресурса или ресурс уже закрыт";
                    case E_API_ERROR_CODES.EC_INVALID_RESOURCE:
                        return "Указан дескриптор несовместимого ресурса";
                    case E_API_ERROR_CODES.EC_INVALID_CONNECTION_TYPE:
                        return "Указан неподдерживаемый тип соединения";
                    case E_API_ERROR_CODES.EC_INVALID_CONNECTION_ADDRESS:
                        return "Адрес подключения не соответствует типу подключения";
                    case E_API_ERROR_CODES.EC_INVALID_DEVICE_ADDRESS:
                        return "Недопустимый адрес устройства";
                    case E_API_ERROR_CODES.EC_DEVICE_OPERATION_UNSUPPORTED:
                        return "Функциональность не поддерживается устройством";
                    case E_API_ERROR_CODES.EC_DEVICE_NOT_CONNECTED:
                        return "Устройство было отключено";
                    case E_API_ERROR_CODES.EC_DEVICE_NO_RESPOND:
                        return "Устройство не отвечает на запросы";
                    case E_API_ERROR_CODES.EC_DEVICE_COMM_FAILURE:
                        return "Ошибика обмена данными с устройством";
                    case E_API_ERROR_CODES.EC_DEVICE_PROTOCOL_FAILURE:
                        return "Ошибка при обработке сообщений протокола";
                    case E_API_ERROR_CODES.EC_POLL_NO_EVENTS:
                        return "Очередь событий пуста";
                    case E_API_ERROR_CODES.EC_POLL_QUEUE_CLOSED:
                        return "Очередь событий закрыта";
                    case E_API_ERROR_CODES.EC_CALL_INIT:
                        return "Требуется инициализация устройства";
                    case E_API_ERROR_CODES.EC_DEVICE_INVALID_COMMAND:
                        return "Нет такой команды или команда не поддерживается устройством";
                    case E_API_ERROR_CODES.EC_DEVICE_INVALID_PARAM:
                        return "Указан недопустимый параметр команды";
                    case E_API_ERROR_CODES.EC_DEVICE_INVALID_PIN:
                        return "Указан неверный PIN для команд настройки";
                    case E_API_ERROR_CODES.EC_DEVICE_COMMAND_TIMEOUT:
                        return "Истекло время ожидания на получение устройством нужной команды";
                    case E_API_ERROR_CODES.EC_DEVICE_NO_CARD:
                        return "Нет карты в поле";
                    case E_API_ERROR_CODES.EC_DEVICE_UNKNOWN_CARD:
                        return "Устройство не смогло распознать карту";
                    case E_API_ERROR_CODES.EC_DEVICE_INCOMPATIBLE_CARD:
                        return "Карта несовместима с выполняемой операцией";
                    case E_API_ERROR_CODES.EC_DEVICE_AUTH_FAIL:
                        return "Ошибка авторизации карты";
                    case E_API_ERROR_CODES.EC_DEVICE_PROFILE_FAIL:
                        return "Используется не подходящий проифль";
                    case E_API_ERROR_CODES.EC_DEVICE_RW_FAIL:
                        return "Ошибка авторизации карты при записи/чтении";
                    case E_API_ERROR_CODES.EC_IO_OPEN_FAIL:
                        return "Ошибка при открытии соединения";
                    case E_API_ERROR_CODES.EC_IO_CLOSE_FAIL:
                        return "Ошибка при закрытии соединения";
                    case E_API_ERROR_CODES.EC_IO_READ_FAIL:
                        return "Ошибка при чтении данных";
                    case E_API_ERROR_CODES.EC_IO_WRITE_FAIL:
                        return "Ошибка при отправке данных";
                    case E_API_ERROR_CODES.EC_IO_CLOSED:
                        return "Операция прервана, соединение было закрыто";
                    default:
                        throw new ArgumentOutOfRangeException(nameof(errorCode), errorCode, null);
                }
            }

            return $"Неизвестная ошибка с кодом {errorCode}";
        }
    }
}