﻿using System;
using Rglib.Interop;

namespace rglib.libtest {
    /// <summary>
    /// Тип профиля
    /// </summary>
    public enum ProfileType {
        /// <summary>
        /// Профиль для работы с Pay
        /// </summary>
        APPLE_GOOGLE_PAY,

        /// <summary>
        /// Профиль для работы с мобильным приложением
        /// </summary>
        MOBILE_ACCESS,

        /// <summary>
        /// Профиль Mifare
        /// </summary>
        MIFARE
    }

    public enum MifareAuthType
    {
        SL1_CLASSIC,
        SL3_PLUS 
    }

    /// <summary>
    /// Профиль
    /// </summary>
    public class MifareProfile {
        /// <summary>
        /// Имя профиля
        /// </summary>
        private string _name;

        public MifareProfile() {
            ProfileType = ProfileType.MIFARE;
            MifareAuthType = MifareAuthType.SL1_CLASSIC;
            ClassicKey = new byte[6] {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
            PlusKey = new byte[16]
                {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
            BlockNumber = 1;
        }

        public string Name {
            get => _name;
            set {
                if (!string.IsNullOrEmpty(value) && !string.IsNullOrWhiteSpace(value))
                    _name = value;
            }
        }

        public ProfileType ProfileType { get; set; }

        public MifareAuthType MifareAuthType { get; set; }

        public bool UseClassicKeyB { get; set; }

        public bool UseGeneratedKey { get; set; }

        public bool UsePresentKey { get; set; }

        public bool UseEmitedKey { get; set; }

        public bool UsePlusKeyB { get; set; }

        public byte[] ClassicKey { get; set; }

        public byte[] PlusKey { get; set; }

        public byte BlockNumber { get; set; }

        public byte FillAccessFlags()
        {
            RG_CARD_AUTH_FLAGS flags = 0;
            if (ProfileType != ProfileType.MIFARE)
            {
                flags |= ProfileType == ProfileType.APPLE_GOOGLE_PAY
                    ? RG_CARD_AUTH_FLAGS.CAF_USE_PAY
                    : RG_CARD_AUTH_FLAGS.CAF_USE_APP;
            }

            if (ProfileType == ProfileType.MIFARE)
            {
                flags |= UseClassicKeyB ? RG_CARD_AUTH_FLAGS.CAF_CLASSIC_KEY_B : 0;
                flags |= UsePlusKeyB ? RG_CARD_AUTH_FLAGS.CAF_PLUS_KEY_B : 0;
                flags |= MifareAuthType == MifareAuthType.SL3_PLUS ? RG_CARD_AUTH_FLAGS.CAF_PLUS_SL3 : 0;
            }
            else
            {
                flags |= UseGeneratedKey ? RG_CARD_AUTH_FLAGS.CAF_GENERATED_KEY : 0;
                flags |= UsePresentKey ? RG_CARD_AUTH_FLAGS.CAF_PRESENT_KEY : 0;
                flags |= UseEmitedKey ? RG_CARD_AUTH_FLAGS.CAF_EMITED_KEY : 0;
            }

            return (byte) flags;
        }

        public override string ToString() {
            switch (ProfileType) {
                case ProfileType.APPLE_GOOGLE_PAY:
                    return $"{Name}: Apple/Google Pay";
                case ProfileType.MOBILE_ACCESS:
                    return $"{Name}: Мобильное приложение";
                case ProfileType.MIFARE:
                    return
                        $"{Name}: Mifare Блок№:{BlockNumber} {(UseClassicKeyB ? "B" : "A")}/{(UsePlusKeyB ? "B" : "A")}";
            }

            return "Неизвестный тип профиля";
        }


    }
}
