﻿using System;
using Rglib.Interop;

namespace rglib.libtest {
    /// <summary>
    /// Считка
    /// </summary>
    public class Reader {
        private readonly RG_DEVICE_INFO_EXT _infoExt;

        public Reader(RG_ENDPOINT readerEndpoint, RG_DEVICE_INFO_EXT infoExt) {
            Endpoint = readerEndpoint;
            _infoExt = infoExt;
        }

        /// <summary>
        /// Конечная точка используемая сичткой
        /// </summary>
        public RG_ENDPOINT Endpoint { get; }

        /// <summary>
        /// Адрес устрйоства
        /// </summary>
        public byte Address => _infoExt.Address;

        /// <summary>
        /// Тип устройства
        /// </summary>
        public RG_DEVICE_TYPE Type {
            get {
                if (Enum.IsDefined(typeof(RG_DEVICE_TYPE), _infoExt.Type))
                    return (RG_DEVICE_TYPE) _infoExt.Type;
                return RG_DEVICE_TYPE.DTE_UNKNOWN;
            }
        }

        public uint SerialNumber => _infoExt.Serial;

        public string FirmwareVersion => $"{(_infoExt.Firmware >> 8 & 0xFF)}.{_infoExt.Firmware & 0xFF}";

        public string EndPointTypeString => Endpoint.Type.ToString();

        public string EndPointAddress => Endpoint.Address;

        public bool CheckCapability(_E_RG_CAPABILITIES cap) {
            return (_infoExt.Capabilities & (uint)cap) > 0;
        }

        /// <summary>
        /// Имя устройства исходя из типа устройства
        /// </summary>
        public string Name {
            get {
                switch (Type) {
                    case RG_DEVICE_TYPE.DTE_RDR202:
                        return "RDR-202";
                    case RG_DEVICE_TYPE.DTE_R10EHT:
                        return "R10-EHT";
                    case RG_DEVICE_TYPE.DTE_R10MF:
                        return "R10-MF";
                    case RG_DEVICE_TYPE.DTE_R15MULTI:
                        return "R15-Multi";
                    case RG_DEVICE_TYPE.DTE_R5USBMULTI:
                        return "R5-USB Multi";
                    case RG_DEVICE_TYPE.DRTE_R5USBMULTIPROF:
                        return "R5-USB Multi Prof";
                    case RG_DEVICE_TYPE.DTE_RDR204EH:
                        return "RDR204-EH Key";
                    case RG_DEVICE_TYPE.DTE_RDR204MF:
                        return "RDR204-MF Key";
                    case RG_DEVICE_TYPE.DTE_RDR102:
                        return "RDR-102";
                    case RG_DEVICE_TYPE.DTE_UNKNOWN:
                    default:
                        return "Неизвестный тип устройства";
                }
            }
        }

        public override string ToString() {
            return $"{Endpoint.Type.ToString()} \"{Endpoint.Address}\" {Name} [{Address}]";
        }

    }
}